<?php

namespace App\Services;

use App\Http\Requests\PerbaikanHakiRequest;
use App\Models\PengajuanHaki;
use App\Models\PerbaikanHaki;
use App\Models\PerbaikanHakiFilePendukung;
use App\Models\PerbaikanHakiKolabolator;
use Illuminate\Support\Facades\DB;

class PerbaikanHakiService
{
    public function copyFromPengajuan(int $idPengajuan): int
    {
        return DB::transaction(function () use ($idPengajuan) {

            /* ===============================
               1. COPY MASTER
            =============================== */
            $pengajuan = PengajuanHaki::findOrFail($idPengajuan);

            $data = $pengajuan->toArray();
            unset($data['id'], $data['created_at'], $data['updated_at']);

            // copy file surat
            if ($pengajuan->file_surat_pengajuan) {
                $data['file_surat_pengajuan'] = $this->copyFile(
                    'file_surat_pengajuan',
                    'perbaikan/file_surat_pengajuan',
                    $pengajuan->file_surat_pengajuan
                );
            }

            $data['id_pengajuan_haki'] = $pengajuan->id;

            $perbaikan = PerbaikanHaki::create($data);

            /* ===============================
               2. COPY FILE PENDUKUNG
            =============================== */
            foreach ($pengajuan->file_pendukung as $file) {
                PerbaikanHakiFilePendukung::create([
                    'id_perbaikan_haki' => $perbaikan->id,
                    'judul' => $file->judul,
                    'file' => $file->file
                        ? $this->copyFile(
                            'file_pendukung',
                            'perbaikan/file_pendukung',
                            $file->file
                        )
                        : null
                ]);
            }

            /* ===============================
               3. COPY KOLABORATOR
            =============================== */
            foreach ($pengajuan->kolabolator as $kolab) {
                PerbaikanHakiKolabolator::create([
                    'id_perbaikan_haki' => $perbaikan->id,
                    'jenis' => $kolab->jenis,
                    'nama' => $kolab->nama,
                    'instansi' => $kolab->instansi,
                    'file_ktp' => $kolab->file_ktp
                        ? $this->copyFile(
                            "file_ktp_kolabolator_" . strtolower($kolab->jenis),
                            "perbaikan/file_ktp_kolabolator_" . strtolower($kolab->jenis),
                            $kolab->file_ktp
                        )
                        : null,
                    'file_npwp' => $kolab->file_npwp
                        ? $this->copyFile(
                            "file_npwp_kolabolator_" . strtolower($kolab->jenis),
                            "perbaikan/file_npwp_kolabolator_" . strtolower($kolab->jenis),
                            $kolab->file_npwp
                        )
                        : null,
                ]);
            }

             /* ===============================
               RETURN ID PERBAIKAN
            =============================== */
            return $perbaikan->id;
        });
    }

    /* ===============================
       FILE COPY HELPER
    =============================== */
    private function copyFile(string $fromDir, string $toDir, string $filename): string
    {
        $source = public_path("$fromDir/$filename");
        if (!file_exists($source)) {
            throw new \Exception("File tidak ditemukan: $source");
        }

        if (!file_exists(public_path($toDir))) {
            mkdir(public_path($toDir), 0777, true);
        }

        $newName = uniqid() . '_' . $filename;
        copy($source, public_path("$toDir/$newName"));

        return $newName;
    }

    /* ==================================================
       UPDATE DATA PERBAIKAN
    ================================================== */
    public function update(PerbaikanHakiRequest $req): void
    {
        DB::transaction(function () use ($req) {

            $perbaikan = PerbaikanHaki::findOrFail($req->id);

            /* ===============================
               UPDATE
            =============================== */
            $perbaikan->fill($req->only([
                'no_surat_pengajuan',
                'email',
                'telp',
                'id_jenis_haki',
                'id_satuan_kerja',
                'judul_kekayaan_intelektual',
                'abstrak_kekayaan_intelektual'
            ]));

            /* ===============================
            DATA BERDASARKAN JENIS HAKI
            =============================== */
            $this->handleJenisHaki($perbaikan, $req);

            /* ===============================
               FILE SURAT (REPLACE)
            =============================== */

            if ($req->hasFile('file_surat_pengajuan')) {
                $this->deleteFile(
                    'perbaikan/file_surat_pengajuan',
                    $perbaikan->file_surat_pengajuan
                );

                $perbaikan->file_surat_pengajuan = $this->upload(
                    $req->file('file_surat_pengajuan'),
                    'perbaikan/file_surat_pengajuan'
                );
            }

            $perbaikan->save();

            /* ===============================
               FILE PENDUKUNG
            =============================== */
            $this->updateFilePendukung($perbaikan, $req);

            /* ===============================
               KOLABORATOR
            =============================== */
            $this->updateKolaborator($perbaikan, $req, 'Internal');
            $this->updateKolaborator($perbaikan, $req, 'Eksternal');
        });
    }

    private function handleJenisHaki($pengajuan, PerbaikanHakiRequest $req)
    {
        switch ($req->id_jenis_haki) {
            case 1: // Merek
                $pengajuan->fill($req->only([
                    'uraian_warna_merk',
                    'arti_merk',
                    'kuasa_merk'
                ]));
                break;

            case 2: // Paten
                $pengajuan->fill($req->only([
                    'id_jenis_paten',
                    'id_bidang_teknologi'
                ]));
                break;

            case 4: // PVT
                $pengajuan->fill($req->only([
                    'judul_pvt',
                    'nama_umum_pvt',
                    'nama_usul_pvt',
                    'negara_asal_pvt',
                    'informasi_teknis_pvt'
                ]));
                break;

            case 5: // Hak Cipta
                $pengajuan->fill($req->only([
                    'tempat_hak_cipta',
                    'tanggal_pengumuman_hak_cipta',
                    'id_jenis_paten',
                    'id_bidang_teknologi'
                ]));
                break;

            case 7: // Indikasi Geografis
                $pengajuan->fill($req->only([
                    'nama_indikasi_geografis',
                    'nama_barang_indikasi_geografis',
                    'kualitas_indikasi_geografis',
                    'karakteristik_indikasi_geografis',
                    'kelas_nice_indikasi_geografis',
                    'sejarah_indikasi_geografis',
                    'tradisi_indikasi_geografis',
                    'jumlah_lisensi_indikasi_geografis'
                ]));
                break;

            case 8: // DTLST
                $pengajuan->fill($req->only([
                    'tanggal_pertama_dientri_dtlst',
                    'jumlah_lisensi_dtlst'
                ]));
                break;
        }
    }

    private function upload($file, string $path): string
    {
        $name = uniqid() . '.' . $file->extension();
        $file->move(public_path($path), $name);
        return $name;
    }

    public function deleteFile(string $path, ?string $filename): void
    {
        if ($filename && file_exists(public_path("$path/$filename"))) {
            unlink(public_path("$path/$filename"));
        }
    }

    /* ==================================================
       UPDATE FILE PENDUKUNG
    ================================================== */
    private function updateFilePendukung($perbaikan, $req): void
    {
        // hapus data & file lama
        // foreach ($perbaikan->filePendukung as $old) {
        //     $this->deleteFile('perbaikan/file_pendukung', $old->file);
        //     $old->delete();
        // }

        if (!$req->judul_file_pendukung) return;

        foreach ($req->judul_file_pendukung as $i => $judul) {

            if(!empty($judul))
            {
                PerbaikanHakiFilePendukung::create([
                    'id_perbaikan_haki' => $perbaikan->id,
                    'judul' => $judul,
                    'file' => !empty($req->file_pendukung[$i])
                        ? $this->upload(
                            $req->file_pendukung[$i],
                            'perbaikan/file_pendukung'
                        )
                        : null
                ]);
            }
        }
    }

    /* ==================================================
       UPDATE KOLABORATOR
    ================================================== */
    private function updateKolaborator($perbaikan, $req, string $jenis): void
    {
        $prefix = strtolower($jenis);

        // hapus lama + file
        // $oldData = $perbaikan->kolaborator()->where('jenis', $jenis)->get();
        // foreach ($oldData as $old) {
        //     $this->deleteFile("perbaikan/file_ktp_kolabolator_$prefix", $old->file_ktp);
        //     $this->deleteFile("perbaikan/file_npwp_kolabolator_$prefix", $old->file_npwp);
        //     $old->delete();
        // }

        $namaList = $req->input("nama_kolabolator_$prefix");
        if (!$namaList) return;

        foreach ($namaList as $i => $nama) {

            if(!empty($nama))
            {
                PerbaikanHakiKolabolator::create([
                    'id_perbaikan_haki' => $perbaikan->id,
                    'jenis' => $jenis,
                    'nama' => $nama,
                    'instansi' => $req->input("instansi_kolabolator_$prefix")[$i] ?? null,
                    'file_ktp' => !empty($req->file("file_ktp_kolabolator_$prefix")[$i])
                        ? $this->upload(
                            $req->file("file_ktp_kolabolator_$prefix")[$i],
                            "perbaikan/file_ktp_kolabolator_$prefix"
                        )
                        : null,
                    'file_npwp' => !empty($req->file("file_npwp_kolabolator_$prefix")[$i])
                        ? $this->upload(
                            $req->file("file_npwp_kolabolator_$prefix")[$i],
                            "perbaikan/file_npwp_kolabolator_$prefix"
                        )
                        : null,
                ]);
            }
        }
    }
}
