<?php

namespace App\Services;

use App\Http\Requests\PengajuanHakiRequest;
use App\Models\PengajuanHaki;
use App\Models\PengajuanHakiFilePendukung;
use App\Models\PengajuanHakiKolabolator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PengajuanHakiService
{
    public function save(PengajuanHakiRequest $req)
    {
        DB::transaction(function () use ($req) {

            $pengajuan = $this->storePengajuan($req);
            $this->storeFilePendukung($pengajuan, $req);
            $this->storeKolaborator($pengajuan, $req, 'Internal');
            $this->storeKolaborator($pengajuan, $req, 'Eksternal');

        });
    }

    /* ===============================
       CREATE / UPDATE UTAMA
    =============================== */
    private function storePengajuan($req)
    {
        $pengajuan = $req->id
            ? PengajuanHaki::findOrFail($req->id)
            : new PengajuanHaki([
                'kode_pengajuan' => PengajuanHaki::no_reg(),
                'id_user' => Auth::id(),
                'status' => 'Review'
            ]);

        $pengajuan->fill($req->only([
            'no_surat_pengajuan',
            'email',
            'telp',
            'id_jenis_haki',
            'id_satuan_kerja',
            'judul_kekayaan_intelektual',
            'abstrak_kekayaan_intelektual'
        ]));

        /* ===============================
        DATA BERDASARKAN JENIS HAKI
        =============================== */
        $this->handleJenisHaki($pengajuan, $req);

        // Update file surat (hapus lama)
        if ($req->hasFile('file_surat_pengajuan')) {
            $this->deleteFile('file_surat_pengajuan', $pengajuan->file_surat_pengajuan);
            $pengajuan->file_surat_pengajuan =
                $this->upload($req->file('file_surat_pengajuan'), 'file_surat_pengajuan');
        }

        $pengajuan->save();

        return $pengajuan;
    }

    /* ===============================
       FILE PENDUKUNG
    =============================== */
    private function storeFilePendukung($pengajuan, $req)
    {
        if (!$req->judul_file_pendukung) return;

        // Optional: delete lama jika update
        // PengajuanHakiFilePendukung::where('id_pengajuan_haki', $pengajuan->id)->delete();

        foreach ($req->judul_file_pendukung as $i => $judul) {

            if(!empty($judul))
            {
                $item = new PengajuanHakiFilePendukung();
                $item->id_pengajuan_haki = $pengajuan->id;
                $item->judul = $judul;

                if (!empty($req->file_pendukung[$i])) {
                    $item->file = $this->upload($req->file_pendukung[$i], 'file_pendukung');
                }

                $item->save();
            }
        }
    }

    /* ===============================
       KOLABORATOR
    =============================== */
    private function storeKolaborator($pengajuan, $req, $jenis)
    {
        $prefix = strtolower($jenis);

        $namaList = $req->input("nama_kolabolator_{$prefix}");
        if (!$namaList) return;

        // PengajuanHakiKolabolator::where([
        //     'id_pengajuan_haki' => $pengajuan->id,
        //     'jenis' => $jenis
        // ])->delete();

        foreach ($namaList as $i => $nama) {

            if(!empty($nama))
            {
                $kolab = new PengajuanHakiKolabolator();
                $kolab->id_pengajuan_haki = $pengajuan->id;
                $kolab->jenis = $jenis;
                $kolab->nama = $nama;
                $kolab->instansi = $req->input("instansi_kolabolator_{$prefix}")[$i] ?? null;

                if (!empty($req->file("file_ktp_kolabolator_{$prefix}")[$i])) {
                    $kolab->file_ktp = $this->upload(
                        $req->file("file_ktp_kolabolator_{$prefix}")[$i],
                        "file_ktp_kolabolator_{$prefix}"
                    );
                }

                if (!empty($req->file("file_npwp_kolabolator_{$prefix}")[$i])) {
                    $kolab->file_npwp = $this->upload(
                        $req->file("file_npwp_kolabolator_{$prefix}")[$i],
                        "file_npwp_kolabolator_{$prefix}"
                    );
                }

                $kolab->save();
            }
        }
    }

    /* ===============================
       FILE HELPER
    =============================== */
    private function upload($file, $path)
    {
        $name = uniqid() . '.' . $file->extension();
        $file->move($path, $name);
        return $name;
    }

    public function deleteFile($path, $filename)
    {
        if ($filename && file_exists(public_path("$path/$filename"))) {
            unlink(public_path("$path/$filename"));
        }
    }

    private function handleJenisHaki($pengajuan, PengajuanHakiRequest $req)
    {
        switch ($req->id_jenis_haki) {
            case 1: // Merek
                $pengajuan->fill($req->only([
                    'uraian_warna_merk',
                    'arti_merk',
                    'kuasa_merk'
                ]));
                break;

            case 2: // Paten
                $pengajuan->fill($req->only([
                    'id_jenis_paten',
                    'id_bidang_teknologi'
                ]));
                break;

            case 4: // PVT
                $pengajuan->fill($req->only([
                    'judul_pvt',
                    'nama_umum_pvt',
                    'nama_usul_pvt',
                    'negara_asal_pvt',
                    'informasi_teknis_pvt'
                ]));
                break;

            case 5: // Hak Cipta
                $pengajuan->fill($req->only([
                    'tempat_hak_cipta',
                    'tanggal_pengumuman_hak_cipta',
                    'id_jenis_paten',
                    'id_bidang_teknologi'
                ]));
                break;

            case 7: // Indikasi Geografis
                $pengajuan->fill($req->only([
                    'nama_indikasi_geografis',
                    'nama_barang_indikasi_geografis',
                    'kualitas_indikasi_geografis',
                    'karakteristik_indikasi_geografis',
                    'kelas_nice_indikasi_geografis',
                    'sejarah_indikasi_geografis',
                    'tradisi_indikasi_geografis',
                    'jumlah_lisensi_indikasi_geografis'
                ]));
                break;

            case 8: // DTLST
                $pengajuan->fill($req->only([
                    'tanggal_pertama_dientri_dtlst',
                    'jumlah_lisensi_dtlst'
                ]));
                break;
        }
    }

    public function delete($id)
    {
        PengajuanHaki::find($id)->delete();
    }
}
